/** @file   bmfont.cpp
 * @brief   Implementation of BMFont class.
 * @version $Revision: 1.2 $
 * @author  Tomi Lamminsaari
 */

#include "bmfont.h"
#include "datatypes.h"
#include <sstream>

using std::string;
using std::map;
using std::istringstream;


namespace eng2d {


///
/// Static members, constants and datatypes
/// =======================================




///
/// Constructors, destructor and operators
/// ======================================
/** Constructor
 */
BMFont::BMFont()
{
}



/** Constructor
 */
BMFont::BMFont( const string& fontname ) throw (xIoError)
{
  if ( this->open( fontname ) != KErrNone ) {
    throw xIoError( "BMFont",
                    "BMFont(const string&)",
                    "Failed to open fontfile." );
  }
}



/** Destructor
 */
BMFont::~BMFont()
{
  this->close();
}




///
/// Public methods
/// ==============

/** Opens new font
 */
int BMFont::open( const string& fontname )
{
  this->close();
  
  DATAFILE* pD = load_datafile( fontname.c_str() );
  if ( pD == 0 ) {
    return KErrNotFound;
  }
  string desc = string( reinterpret_cast<char*>( pD[0].dat ) );
  istringstream ss( desc );
  
  string tmp;
  while ( true ) {
    if ( ss.eof() == true ) {
      unload_datafile( pD );
      return KErrEof;
    }
    
    ss >> tmp;
    if ( tmp == "<eng2d_bitmapfont_v1>" ) {
      break;
    }
  }
  
  int index = 1;
  while ( true ) {
    if ( ss.eof() == true ) {
      unload_datafile( pD );
      return KErrEof;
    }
    
    ss >> tmp;
    if ( tmp == "#" ) {
      ss.ignore( 4096, '\n' );
      
    } else if ( tmp == "<font>" ) {
      BITMAP* pBM = reinterpret_cast<BITMAP*>( pD[index].dat );
      index += 1;
      GlyphSet* pG = new GlyphSet( pBM, ss );
      if ( pG->bad() == true ) {
        delete pG;
        pG = 0;
      }
      // Constructortion and initialization of the glyphset was successful so
      // add it to the glyphtable.
      m_glyphsets[ pG->id() ] = pG;
      
    } else if ( tmp == "name:" ) {
      ss >> m_name;
      
    } else if ( tmp == "</eng2d_bitmapfont_v1>" ) {
      break;
      
    }
  }
  unload_datafile( pD );
  return KErrNone;
}



/** Closes the current font
 */
void BMFont::close()
{
  if ( m_glyphsets.size() > 0 ) {
    map<string, GlyphSet*>::const_iterator it = m_glyphsets.begin();
    while ( it != m_glyphsets.end() ) {
      delete it->second;
      it++;
    }
  }
  m_glyphsets.clear();
}





///
/// Getter methods
/// ==============

/** Tells if there if data for font with given size.
 */
bool BMFont::hasSize( int s ) const
{
  map<string, GlyphSet*>::const_iterator it = m_glyphsets.begin();
  while ( it != m_glyphsets.end() ) {
    if ( it->second != 0 ) {
      if ( it->second->glyphSize() == s ) {
        return true;
      }
    }
    it++;
  }
  return false;
}



/** Returns the glyphset for given sized font
 */
const GlyphSet* BMFont::glyphset( int s ) const
{
  map<string, GlyphSet*>::const_iterator it = m_glyphsets.begin();
  while ( it != m_glyphsets.end() ) {
    if ( it->second != 0 ) {
      if ( it->second->glyphSize() == s ) {
        return it->second;
      }
    }
    it++;
  }
  return 0;
}



/** Returns the glyphset with given id.
 */
const GlyphSet* BMFont::glyphset( const string& id ) const
{
  map<string, GlyphSet*>::const_iterator it = m_glyphsets.find( id );
  if ( it == m_glyphsets.end() ) {
    return 0;
  }
  return it->second;
}




///
/// Private or Protected methods
/// ============================


} // end of namespace
